/*
AnimationFactory
Copyright (C) 2003 Bj\"orn Hoffmeister

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package de.uni_luebeck.tcs.animation;

import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import javax.swing.*;

public class AnimationFactory {
    final int DEFAULT_TIME = 10000;
    final int DEFAULT_SPEED = 512;


    static abstract class LabelAnimation implements Animator.Animation {
	protected static final int SHADOW_DIV = 3;
	protected static final int Z_LEVEL = 10;
	protected static final int FONT_STEP = 2;
	protected static final int Z_STEP = 6;
	protected static final int SHADOW_ALPHA = 64;
	protected static final Color SHADOW_CLR = new Color(0, 0, 0, SHADOW_ALPHA);

	protected static class FlyingLabel {
	    // Label, Schatten und gemorphtes Label
	    final JLabel jlbClone, jlbShadow, jlbMorph;
	    // Start- und Endpunkt
	    final int intX0, intY0,  intX1, intY1;
	    // x-, y-, euklidische-Distanz
	    final int intDX, intDY, intD;
	    // Flugzeit
	    final int intTime;

	    public FlyingLabel(
		JLabel jlbClone, JLabel jlbShadow, JLabel jlbMorph,
		int intX0, int intY0, int intX1, int intY1,
		int intTime
	    ) {
		this.jlbClone = jlbClone;
		this.jlbShadow = jlbShadow;
		this.jlbMorph = jlbMorph;
		this.intX0 = intX0;
		this.intY0 = intY0;
		this.intX1 = intX1;
		this.intY1 = intY1;
		this.intDX = intX1 - intX0;
		this.intDY = intY1 - intY0;
		double dblDX = (double)intDX;
		double dblDY = (double)intDY;
		this.intD = (int)StrictMath.sqrt(dblDX * dblDX + dblDY * dblDY);
		this.intTime = intTime;
	    }
	}

	protected final FlyingLabel[] flyLabel;
	protected final boolean[] blnDown;
	protected final JLabel[] jlbSource;
	protected final JLabel jlbTarget;
	protected final Color clrTarget;
	protected final int[] intShadowOffset;
	protected final int intOffset;

	protected final Container cntAnimation;

	protected int intFlying;
	protected boolean blnWaiting = true;
	protected boolean blnIsFirst = true;
	protected boolean blnFinnished = false;

	public LabelAnimation(JLabel[] jlbSource, JLabel jlbTarget, Color clrTarget, int intOffset, Animator anmThis) {
	    this.cntAnimation = anmThis.getAnimationPane();
	    this.jlbSource = jlbSource;
	    this.jlbTarget = jlbTarget;
	    this.clrTarget = clrTarget;

	    this.flyLabel = new FlyingLabel[jlbSource.length];
	    this.blnDown = new boolean[jlbSource.length];

	    this.intShadowOffset = new int[Z_LEVEL];	    

	    this.intOffset = intOffset;
	}
	
	protected final Point getTargetIn(Component cmpTarget, Component cmpParent) {
	    return SwingUtilities.convertPoint(cmpTarget, 0, 0, cmpParent);
	}

	public void init() {
	    for (int i= 0; i < flyLabel.length; i++) {
		if (flyLabel[i].jlbClone != null) {
		    flyLabel[i].jlbClone.setVisible(false);
		    cntAnimation.add(flyLabel[i].jlbClone);
		}
		if (flyLabel[i].jlbShadow != null) {
		    flyLabel[i].jlbShadow.setVisible(false);
		    cntAnimation.add(flyLabel[i].jlbShadow);
		}
		if (flyLabel[i].jlbMorph != null) {
		    flyLabel[i].jlbMorph.setVisible(false);
		    cntAnimation.add(flyLabel[i].jlbMorph);
		}
	    }
	    Arrays.fill(blnDown, false);
	    this.intFlying = flyLabel.length;

	    if (intOffset <= 0) {
		this.blnWaiting = false;
		start();
	    }
	}

	protected void start() {
	    for (int i = 0; i < jlbSource.length; i++) {
		jlbSource[i].setForeground(clrTarget);
		//		jlbSource[i].repaint();
	    }
	    for (int i = 0; i < flyLabel.length; i++) {
		if (flyLabel[i].jlbClone != null) flyLabel[i].jlbClone.setVisible(true);
		if (flyLabel[i].jlbShadow != null) flyLabel[i].jlbShadow.setVisible(true);
		if (flyLabel[i].jlbMorph != null) flyLabel[i].jlbMorph.setVisible(true);
	    }	    
	}
	public void stop() {
	    blnFinnished = true;
	    for (int i = 0; i < jlbSource.length; i++) {
		jlbSource[i].setForeground(clrTarget);
		//		jlbSource[i].repaint();
	    }
	    jlbTarget.setForeground(clrTarget);
	    //	    jlbTarget.repaint();
	}

	public boolean isFinnished() {
	    return blnFinnished;
	}
	
	protected final int onAir(int intTimePassed) {
	    if (blnFinnished) return -1;

	    if (blnWaiting) {
		if (intTimePassed < intOffset) return -1;
		start();
		blnWaiting = false;
	    }
	    return intTimePassed - intOffset;	    
	}

	/*
	public Dimension getRelativeSpeed(int intDX, int intDY) {
	    int intC = (int)Math.sqrt(intDX * intDX + intDY * intDY) / SPEED;
	    return new Dimension(intDX / intC, intDY / intC);	    
	}

	public Dimension getSpeedForTime(int intDX, int intDY) {
	    return new Dimension((1000 * intDX) / TIME, (1000 * intDY) / TIME);
	}
	*/
    }



    static class LabelReunification extends LabelAnimation {
	protected final Font[] fntCommon;
	protected final Dimension[] dimPreferredSize;
	protected final int[] intOffsetX, intOffsetY;

	public LabelReunification(JLabel[] jlbSource, JLabel jlbTarget, Color clrTarget, int intOffset, Animator anmThis, int intTime) {
	    super(jlbSource, jlbTarget, clrTarget, intOffset, anmThis);

	    String strLabel = jlbSource[0].getText();

	    // Baue Fonts und Schatten auf den Z-Ebenen
	    this.fntCommon = new Font[Z_LEVEL];
	    fntCommon[0] = jlbSource[0].getFont();
	    JLabel jlbDummy = new JLabel(strLabel);
	    jlbDummy.setFont(fntCommon[0]);
	    this.dimPreferredSize = new Dimension[Z_LEVEL];
	    dimPreferredSize[0] = jlbDummy.getPreferredSize();
	    this.intOffsetX = new int[Z_LEVEL];
	    intOffsetX[0] = 0;
	    this.intOffsetY = new int[Z_LEVEL];
	    intOffsetY[0] = 0;
	    intShadowOffset[0] = 0;
	    int intSize = fntCommon[0].getSize();
	    for (int i = 1; i < Z_LEVEL; i++) {
		intSize += FONT_STEP;
		fntCommon[i] = new Font(fntCommon[0].getFamily(), fntCommon[0].getStyle(), intSize);
		jlbDummy.setFont(fntCommon[i]);
		dimPreferredSize[i] = jlbDummy.getPreferredSize();
		intOffsetX[i] = (dimPreferredSize[i].width - dimPreferredSize[0].width) / 2;
		intOffsetY[i] = (dimPreferredSize[i].height - dimPreferredSize[0].height) / 2;	       
		intShadowOffset[i] = i / SHADOW_DIV + 1;
	    }

	    // Baue fliegende Klon- und Schatten-Labels
	    Container cntParent = anmThis.getRootPane();
	    Point pntTarget = getTargetIn(jlbTarget, cntParent);
	    for (int i = 0; i < jlbSource.length; i++) {
		Point pntSource = getTargetIn(jlbSource[i], cntParent);

		JLabel jlbClone = new JLabel(strLabel);
		jlbClone.setOpaque(false);
		jlbClone.setForeground(clrTarget);
		jlbClone.setFont(fntCommon[0]);
		jlbClone.setBounds(pntSource.x, pntSource.y, dimPreferredSize[0].width, dimPreferredSize[0].height);

		JLabel jlbShadow = new JLabel(strLabel);
		jlbShadow.setOpaque(false);
		jlbShadow.setForeground(SHADOW_CLR);
		jlbShadow.setFont(fntCommon[0]);
		jlbShadow.setBounds(pntSource.x, pntSource.y, dimPreferredSize[0].width, dimPreferredSize[0].height);
		flyLabel[i] = new FlyingLabel(
		    jlbClone, jlbShadow, null,
		    pntSource.x, pntSource.y, pntTarget.x, pntTarget.y, 
		    intTime
		);	    
	    }
	}

	public void paint(Graphics g, int intTimePassed) {
	    int intFlyTime = onAir(intTimePassed);
	    if (intFlyTime == -1) return;

	    for (int i = 0; i < flyLabel.length; i++) {
		// schon lange angekommen
		if (blnDown[i]) continue;
		// gerade angekommen
		if (intFlyTime > flyLabel[i].intTime) {
		    cntAnimation.remove(flyLabel[i].jlbClone);
		    cntAnimation.remove(flyLabel[i].jlbShadow);
		    flyLabel[i] = null;
		    if (blnIsFirst) {
			blnIsFirst = false;
			jlbTarget.setForeground(clrTarget);
		    }			
		    blnDown[i] = true;
		    intFlying--;
		    continue;
		}
		
		// neue Position
		int intDX = (flyLabel[i].intDX * intFlyTime) / flyLabel[i].intTime;
		int intDY = (flyLabel[i].intDY * intFlyTime) / flyLabel[i].intTime;

		// Start-/Landeanflug
		int intDFrom = (flyLabel[i].intD * intFlyTime) / flyLabel[i].intTime;
		int intDTo = flyLabel[i].intD - intDFrom;		    
		int intDZ = Math.max(Math.min(
		    Math.min(intDFrom / Z_STEP, intDTo / Z_STEP),
		    Z_LEVEL - 1
		    ), 0);
		flyLabel[i].jlbClone.setFont(fntCommon[intDZ]);
		flyLabel[i].jlbShadow.setFont(fntCommon[intDZ]);

		// flieg
		int intX = flyLabel[i].intX0 + intDX - intOffsetX[intDZ];
		int intY = flyLabel[i].intY0 + intDY - intOffsetY[intDZ];
		Dimension dimSize = dimPreferredSize[intDZ];
		flyLabel[i].jlbClone.setBounds(
	            intX, intY, 
		    dimSize.width, dimSize.height
		);		
		flyLabel[i].jlbShadow.setBounds(
		    intX + intShadowOffset[intDZ], intY - intShadowOffset[intDZ],
		    dimSize.width, dimSize.height
		);

		//		System.out.println("x=" + intX + ", y=" + intY + ", width=" + dimSize.width + ", height=" + dimSize.height);
	    }
	    if (intFlying == 0) stop();	    
	}
    }



    static class LabelUnification extends LabelAnimation {
	// Angabe in Prozent der Flugdistanz
	protected static final int MORPH_BEGIN = 28;
	protected static final int MORPH_END = 78;

	protected static final int NOT_STARTED = 0;
	protected static final int PHASE_1 = 1;
	protected static final int PHASE_2 = 2;
	protected static final int PHASE_3 = 3;
	protected static final int FINNISHED = 100;

	protected final Color[] CLONE_CLR;
	protected final Color[] MORPH_CLR;
	protected final Color[] SHADE_CLR;

	protected final Font[] fntCommon;
	protected final Dimension[][] dimPreferredSize;
	protected final int[][] intOffsetX, intOffsetY;

	protected final Font[] fntMorph;
	protected final Dimension[] dimMorphPreferredSize;
	protected final int[] intMorphOffsetX, intMorphOffsetY;

	protected final int[] intMorphBegin;
	protected final int[] intMorphEnd;
	protected final int[] intMorphD;
	protected final int[] intMorphState;

	public LabelUnification(JLabel[] jlbSource, JLabel jlbTarget, Color clrTarget, int intOffset, Animator anmThis, int intTime) {
	    super(jlbSource, jlbTarget, clrTarget, intOffset, anmThis);

	    // Vorausberechnung der benoetigten Farben
	    this.CLONE_CLR = new Color[256];
	    //	    CLONE_CLR[0] = null;
	    //	    CLONE_CLR[255] = jlbSource[0].getForeground();
	    CLONE_CLR[255] = clrTarget;
	    int intR = CLONE_CLR[255].getRed();
	    int intG = CLONE_CLR[255].getGreen();
	    int intB = CLONE_CLR[255].getBlue();
	    for(int i = 0; i < CLONE_CLR.length - 1; i++) CLONE_CLR[i] = new Color(intR, intG, intB, i);

	    this.MORPH_CLR = new Color[256];
	    //	    MORPH_CLR[0] = null;
	    MORPH_CLR[255] = clrTarget;
	    intR = MORPH_CLR[255].getRed();
	    intG = MORPH_CLR[255].getGreen();
	    intB = MORPH_CLR[255].getBlue();
	    for(int i = 0; i < MORPH_CLR.length - 1; i++) MORPH_CLR[i] = new Color(intR, intG, intB, i);

	    this.SHADE_CLR = new Color[SHADOW_ALPHA + 1];
	    //	    SHADE_CLR[0] = null;
	    SHADE_CLR[SHADOW_ALPHA] = SHADOW_CLR;
	    intR = SHADE_CLR[SHADOW_ALPHA].getRed();
	    intG = SHADE_CLR[SHADOW_ALPHA].getGreen();
	    intB = SHADE_CLR[SHADOW_ALPHA].getBlue();
	    for(int i = 0; i < SHADE_CLR.length - 1; i++) SHADE_CLR[i] = new Color(intR, intG, intB, i);

	    String strMorph = jlbTarget.getText();

	    // Baue Source-Fonts, Target-Fonts und Schatten auf den Z-Ebenen
	    // -> nehme ersten Label als Referenz
	    this.fntCommon = new Font[Z_LEVEL];
	    fntCommon[0] = jlbSource[0].getFont();
	    this.fntMorph = new Font[Z_LEVEL];
	    fntMorph[0] = jlbTarget.getFont();

	    JLabel jlbDummyMorph = new JLabel(strMorph);
	    jlbDummyMorph.setFont(fntMorph[0]);
	    this.dimMorphPreferredSize = new Dimension[Z_LEVEL];
	    dimMorphPreferredSize[0] = jlbDummyMorph.getPreferredSize();
	    this.intMorphOffsetX = new int[Z_LEVEL];
	    intMorphOffsetX[0] = 0;
	    this.intMorphOffsetY = new int[Z_LEVEL];
	    intMorphOffsetY[0] = 0;

	    intShadowOffset[0] = 0;
	    int intSize = fntCommon[0].getSize();
	    int intMorphSize = fntMorph[0].getSize();
	    for (int i = 1; i < Z_LEVEL; i++) {
		intSize += FONT_STEP;
		fntCommon[i] = new Font(fntCommon[0].getFamily(), fntCommon[0].getStyle(), intSize);

		intMorphSize += FONT_STEP;
		fntMorph[i] = new Font(fntMorph[0].getFamily(), fntMorph[0].getStyle(), intMorphSize);
		jlbDummyMorph.setFont(fntMorph[i]);
		dimMorphPreferredSize[i] = jlbDummyMorph.getPreferredSize();
		intMorphOffsetX[i] = (dimMorphPreferredSize[i].width - dimMorphPreferredSize[0].width) / 2;
		intMorphOffsetY[i] = (dimMorphPreferredSize[i].height - dimMorphPreferredSize[0].height) / 2;	       

		intShadowOffset[i] = i / SHADOW_DIV + 1;
	    }

	    // Baue fliegende Klon-, Schatten- und Morph-Labels
	    this.dimPreferredSize = new Dimension[jlbSource.length][Z_LEVEL];
	    this.intOffsetX = new int[jlbSource.length][Z_LEVEL];
	    this.intOffsetY = new int[jlbSource.length][Z_LEVEL];
	    this.intMorphBegin = new int[jlbSource.length];
	    this.intMorphEnd = new int[jlbSource.length];
	    this.intMorphD = new int[jlbSource.length];
	    this.intMorphState = new int[jlbSource.length];
	    Arrays.fill(intMorphState, NOT_STARTED);

	    Container cntParent = anmThis.getRootPane();
	    Point pntTarget = getTargetIn(jlbTarget, cntParent);
	    for (int i = 0; i < jlbSource.length; i++) {
		Point pntSource = getTargetIn(jlbSource[i], cntParent);		

		JLabel jlbClone = new JLabel(jlbSource[i].getText());
		jlbClone.setOpaque(false);
		jlbClone.setForeground(CLONE_CLR[255]);
		for (int j = 0; j < Z_LEVEL; j++) {		    
		    jlbClone.setFont(fntCommon[j]);
		    dimPreferredSize[i][j] = jlbClone.getPreferredSize();
		    intOffsetX[i][j] = (dimPreferredSize[i][j].width - dimMorphPreferredSize[0].width) / 2;
		    intOffsetY[i][j] = (dimPreferredSize[i][j].height - dimMorphPreferredSize[0].height) / 2;
		}		    		
		jlbClone.setFont(fntCommon[0]);
		int intX = pntSource.x + intOffsetX[i][0];
		int intY = pntSource.y + intOffsetY[i][0];
		jlbClone.setBounds(pntSource.x, pntSource.y, dimPreferredSize[i][0].width, dimPreferredSize[i][0].height);

		JLabel jlbShadow = new JLabel(jlbSource[i].getText());
		jlbShadow.setOpaque(false);
		jlbShadow.setForeground(SHADOW_CLR);
		jlbShadow.setFont(fntCommon[0]);
		jlbShadow.setBounds(pntSource.x, pntSource.y, dimPreferredSize[i][0].width, dimPreferredSize[i][0].height);

		JLabel jlbMorph = new JLabel(strMorph);
		jlbMorph.setOpaque(false);
		jlbMorph.setForeground(MORPH_CLR[1]);
		jlbMorph.setFont(fntMorph[0]);
		jlbMorph.setVisible(false);

		flyLabel[i] = new FlyingLabel(
		    jlbClone, jlbShadow, jlbMorph,
		    intX, intY, pntTarget.x, pntTarget.y, 
		    intTime
		);

		// set morphing bounds
		intMorphBegin[i] = (flyLabel[i].intD * MORPH_BEGIN) / 100;
		intMorphEnd[i] = (flyLabel[i].intD * MORPH_END) / 100;
		intMorphD[i] = intMorphEnd[i] - intMorphBegin[i];
	    }
	}
	    
	public void paint(Graphics g, int intTimePassed) {
	    int intFlyTime = onAir(intTimePassed);
	    if (intFlyTime == -1) return;

	    for (int i = 0; i < flyLabel.length; i++) {
		// schon lange angekommen
		if (blnDown[i]) continue;
		// gerade angekommen
		if (intFlyTime > flyLabel[i].intTime) {
		    cntAnimation.remove(flyLabel[i].jlbClone);
		    cntAnimation.remove(flyLabel[i].jlbShadow);
		    cntAnimation.remove(flyLabel[i].jlbMorph);
		    flyLabel[i] = null;
		    if (blnIsFirst) {
			blnIsFirst = false;
			jlbTarget.setForeground(clrTarget);
		    }			
		    blnDown[i] = true;
		    intFlying--;
		    continue;
		}
		
		// neue Position
		int intDX = (flyLabel[i].intDX * intFlyTime) / flyLabel[i].intTime;
		int intDY = (flyLabel[i].intDY * intFlyTime) / flyLabel[i].intTime;

		// Start-/Landeanflug
		int intDFrom = (flyLabel[i].intD * intFlyTime) / flyLabel[i].intTime;
		int intDTo = flyLabel[i].intD - intDFrom;		    
		int intDZ = Math.max(Math.min(
		    Math.min(intDFrom / Z_STEP, intDTo / Z_STEP),
		    Z_LEVEL - 1
		), 0);

		// flieg und morph
		if (intDFrom < intMorphBegin[i]) {
		    // Klon
		    int intX = flyLabel[i].intX0 + intDX - intOffsetX[i][intDZ];
		    int intY = flyLabel[i].intY0 + intDY - intOffsetY[i][intDZ];
		    Dimension dimSize = dimPreferredSize[i][intDZ];
		    flyLabel[i].jlbClone.setFont(fntCommon[intDZ]);
		    flyLabel[i].jlbClone.setBounds(
			intX, intY, 
			dimSize.width, dimSize.height
		    );		    
		    // Schatten
		    flyLabel[i].jlbShadow.setFont(fntCommon[intDZ]);
		    flyLabel[i].jlbShadow.setBounds(
			intX + intShadowOffset[intDZ], intY - intShadowOffset[intDZ],
			dimSize.width, dimSize.height
		    );
		} else if (intDFrom > intMorphEnd[i]) {
		    if (intMorphState[i] != FINNISHED) {
			flyLabel[i].jlbClone.setVisible(false);
			flyLabel[i].jlbMorph.setForeground(MORPH_CLR[255]);
			flyLabel[i].jlbMorph.setVisible(true);
			flyLabel[i].jlbShadow.setText(jlbTarget.getText());
			flyLabel[i].jlbShadow.setForeground(SHADOW_CLR);
			flyLabel[i].jlbShadow.setVisible(true);
			intMorphState[i] = FINNISHED;
		    }
		    // Morph
		    int intX = flyLabel[i].intX0 + intDX - intMorphOffsetX[intDZ];
		    int intY = flyLabel[i].intY0 + intDY - intMorphOffsetY[intDZ];
		    Dimension dimSize = dimMorphPreferredSize[intDZ];
		    flyLabel[i].jlbMorph.setFont(fntMorph[intDZ]);
		    flyLabel[i].jlbMorph.setBounds(
			intX, intY, 
			dimSize.width, dimSize.height
		    );		    
		    // Schatten
		    flyLabel[i].jlbShadow.setFont(fntMorph[intDZ]);
		    flyLabel[i].jlbShadow.setBounds(
			intX + intShadowOffset[intDZ], intY - intShadowOffset[intDZ],
			dimSize.width, dimSize.height
		    );
		} else {
		    int intMorphAlpha = StrictMath.min(StrictMath.max(
			(255 * (Math.max(0, intDFrom) - intMorphBegin[i])) / intMorphD[i], 1), 254);
		    int intCloneAlpha = 255 - intMorphAlpha;
		    int intShadowAlpha = 0;
		    
		    switch (intMorphState[i]) {
		    case NOT_STARTED:
			flyLabel[i].jlbMorph.setVisible(true);
			intMorphState[i] = PHASE_1;
		    case PHASE_1:
			int intShadowAlpha1 = SHADOW_ALPHA - intMorphAlpha / 2;
			int intShadowAlpha2 = SHADOW_ALPHA - intCloneAlpha / 2;
			if ((intShadowAlpha2 > intShadowAlpha1) || (1 > intShadowAlpha1)) {
			    flyLabel[i].jlbShadow.setVisible(false);
			    intMorphState[i] = PHASE_2;
			} else {
			    intShadowAlpha = intShadowAlpha1;
			    break;
			}		    
		    case PHASE_2:
			intShadowAlpha = SHADOW_ALPHA - intCloneAlpha / 2;
			if (intShadowAlpha > 0) {			    
			    flyLabel[i].jlbShadow.setText(jlbTarget.getText());
			    flyLabel[i].jlbShadow.setVisible(true);
			    intMorphState[i] = PHASE_3;
			} else {
			    break;
			}    
		    case PHASE_3:
			intShadowAlpha = SHADOW_ALPHA - intCloneAlpha / 2;
			break;
		    }
		    intCloneAlpha = Math.max(0, Math.min(intCloneAlpha, CLONE_CLR.length - 1));
		    intMorphAlpha = Math.max(0, Math.min(intMorphAlpha, MORPH_CLR.length - 1));
		    intShadowAlpha = Math.max(0, Math.min(intShadowAlpha, SHADE_CLR.length -1));

		    // Klon
		    int intX = flyLabel[i].intX0 + intDX - intOffsetX[i][intDZ];
		    int intY = flyLabel[i].intY0 + intDY - intOffsetY[i][intDZ];
		    Dimension dimSize = dimPreferredSize[i][intDZ];		    
		    flyLabel[i].jlbClone.setFont(fntCommon[intDZ]);
		    flyLabel[i].jlbClone.setForeground(CLONE_CLR[intCloneAlpha]);
		    flyLabel[i].jlbClone.setBounds(
			intX, intY, 
			dimSize.width, dimSize.height
		    );

		    // Morph
		    int intMorphX = flyLabel[i].intX0 + intDX - intMorphOffsetX[intDZ];
		    int intMorphY = flyLabel[i].intY0 + intDY - intMorphOffsetY[intDZ];
		    Dimension dimMorphSize = dimMorphPreferredSize[intDZ];
		    flyLabel[i].jlbMorph.setFont(fntMorph[intDZ]);
		    flyLabel[i].jlbMorph.setForeground(MORPH_CLR[intMorphAlpha]);
		    flyLabel[i].jlbMorph.setBounds(
			intMorphX, intMorphY, 
			dimMorphSize.width, dimMorphSize.height
		    );
		    
		    // Schatten
		    switch (intMorphState[i]) {
		    case PHASE_1:
			flyLabel[i].jlbShadow.setFont(fntCommon[intDZ]);
			flyLabel[i].jlbShadow.setForeground(SHADE_CLR[intShadowAlpha]);
			flyLabel[i].jlbShadow.setBounds(
			    intX + intShadowOffset[intDZ], intY - intShadowOffset[intDZ],
			    dimSize.width, dimSize.height
			);
			break;
		    case PHASE_3:
			flyLabel[i].jlbShadow.setFont(fntMorph[intDZ]);
			flyLabel[i].jlbShadow.setForeground(SHADE_CLR[intShadowAlpha]);
			flyLabel[i].jlbShadow.setBounds(
			    intMorphX + intShadowOffset[intDZ], intMorphY - intShadowOffset[intDZ],
			    dimMorphSize.width, dimMorphSize.height
			);
			break;
		    }
		}			    		    		    		 

		//		System.out.println("x=" + intX + ", y=" + intY + ", width=" + dimSize.width + ", height=" + dimSize.height);
	    }
	    if (intFlying == 0) stop();	    
	}
    }

    


    public abstract class StringAnimation implements  Animator.Animation {    
	protected static final int SHADOW_ALPHA = 64;

	protected final JComponent jpcRoot;
	protected final int intRootWidth;
	protected final int intRootHeight;

	protected final int intOffset;
	protected final Color[] clrShadow;
	protected boolean blnFinnished = false;
	
	protected Graphics g = null;
	
	protected StringAnimation(int intOffset) {
	    this.intOffset = intOffset;
	    this.jpcRoot = anmThis.getRootPane();
	    this.intRootWidth = jpcRoot.getWidth();
	    this.intRootHeight = jpcRoot.getHeight();
	    //	    System.out.println("root.w=" + intRootWidth + ", root.h=" + intRootHeight);
	    this.clrShadow = new Color[SHADOW_ALPHA + 1];
	    for (int i = 0; i <= SHADOW_ALPHA; i++) clrShadow[i] = new Color(0, 0, 0, i);		
	}

	// erst hier Komponenten dem AnimationsPane hinzufuegen
	public void init() {
	    blnFinnished = false;
	}

	// persistente Aenderungen vornehmen, die im weiteren Verlauf der
	// Animation aufgetreten waeren
	// Komponenten aus AnimationsPane entfernen ist nicht notwendig
	public void stop() {}

	public void paint(Graphics g, int intRunning) {
	    if ((intRunning < intOffset) || (blnFinnished)) return;

	    final int intTrueRunning  = intRunning - intOffset;
	    if (intTrueRunning >= getRunningTime()) {
		blnFinnished = true;
		return;
	    }

	    this.g = g;
	    paint2(g, intTrueRunning);
	}

	protected abstract void paint2(Graphics g, int intRunning);

	public abstract int getRunningTime();

	public boolean isFinnished() {
	    return blnFinnished;
	}

	protected void drawString(String strDraw, int intX, int intY) {
	    final Color clrTmp = g.getColor();
	    final int intShadow = StrictMath.min(SHADOW_ALPHA - 255 + clrTmp.getAlpha(), clrShadow.length - 1);

	    if (intShadow > 0) {
		final int intOffset = g.getFont().getSize() / 10;
		g.setColor(clrShadow[intShadow]);
		g.drawString(strDraw, intX + intOffset, intY - intOffset);
		g.setColor(clrTmp);
	    }
	    g.drawString(strDraw, intX, intY);	    
	}

	protected Point getCenteredPosition(String strTarget, Font fntTarget) {
	    FontMetrics fmtTarget = jpcRoot.getFontMetrics(fntTarget);
	    return new Point(
		(intRootWidth - fmtTarget.stringWidth(strTarget)) / 2,
		(intRootHeight + fmtTarget.getHeight() / 2) / 2
	    );	
	}

	protected int getOffset(String strTarget, Font fntTarget, int intBegin) {
	    FontMetrics fmtTarget = jpcRoot.getFontMetrics(fntTarget);
	    return fmtTarget.stringWidth(strTarget) - fmtTarget.stringWidth(strTarget.substring(intBegin));
	}
    }


    public class StringGrowInAndOut extends StringAnimation {
	protected static final int GROW_IN_TIME = 3500;
	protected static final int GROW_OUT_TIME = 1500;
	protected static final int WAIT_TIME = 2500;

	protected final int intTime;
	protected final String strGrow;
	protected final Color clrGrow;
	protected final Color[] clrGrowOut;
	protected final Font fntGrow;
	protected final Font[] fntGrowIn;
	protected final Font[] fntGrowOut;
	protected final int intX;
	protected final int intY;
	protected final int intXGrowIn[];
	protected final int intYGrowIn[];
	protected final int intXGrowOut[];
	protected final int intYGrowOut[];
	
	public StringGrowInAndOut(String strGrow, Color clrGrow, Font fntGrow, int intOffset) {
	    super(intOffset);
	    this.intTime = GROW_IN_TIME + WAIT_TIME + GROW_OUT_TIME;
	    this.strGrow = strGrow;
	    this.clrGrow = clrGrow;
	    this.fntGrow = fntGrow;

	    final String strFamily = fntGrow.getFamily();
	    final int intStyle = fntGrow.getStyle();
	    final int intSize = fntGrow.getSize();

	    Point pntXY = getCenteredPosition(strGrow, fntGrow);
	    intX = pntXY.x; intY = pntXY.y;

	    this.fntGrowIn = new Font[intSize];
	    this.intXGrowIn = new int[intSize];
	    this.intYGrowIn = new int[intSize];
	    for (int i = 0; i < intSize; i++) {
		fntGrowIn[i] = new Font(strFamily, intStyle, i);
		pntXY = getCenteredPosition(strGrow, this.fntGrowIn[i]);
		intXGrowIn[i] = pntXY.x; intYGrowIn[i] = pntXY.y;
	    }

	    final int intSizeOut = (3 * intSize) / 2;
	    this.fntGrowOut = new Font[intSizeOut];
	    this.intXGrowOut = new int[intSizeOut];
	    this.intYGrowOut = new int[intSizeOut];
	    for (int i = 0; i < intSizeOut; i++) {
		fntGrowOut[i] = new Font(strFamily, intStyle, intSize + i);
		pntXY = getCenteredPosition(strGrow, this.fntGrowOut[i]);
		intXGrowOut[i] = pntXY.x; intYGrowOut[i] = pntXY.y;
	    }				
	    final int intR = clrGrow.getRed();
	    final int intG = clrGrow.getGreen();
	    final int intB = clrGrow.getBlue();
	    this.clrGrowOut = new Color[256];
	    for (int i = 0; i < 256; i++) this.clrGrowOut[i] = new Color(intR, intG, intB, 255 - i);
	    this.clrGrowOut[255] = clrGrow;
	}

	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    Color clrTmp = g.getColor();
	    Font fntTmp = g.getFont();

	    if (intRunning < GROW_IN_TIME) {
		int intFontIndex = StrictMath.max(StrictMath.min((fntGrowIn.length * intRunning) / GROW_IN_TIME, fntGrowIn.length - 1), 0);
		g.setColor(clrGrow);
		g.setFont(fntGrowIn[intFontIndex]);
		g.drawString(strGrow, intXGrowIn[intFontIndex], intYGrowIn[intFontIndex]);		
	    } else if (intRunning < GROW_IN_TIME + WAIT_TIME) {
		g.setColor(clrGrow);
		g.setFont(fntGrow);	    
		g.drawString(strGrow, intX, intY);		
	    } else {
		intRunning -= (GROW_IN_TIME + WAIT_TIME);
		int intColorIndex = StrictMath.max(StrictMath.min((clrGrowOut.length * intRunning) / GROW_OUT_TIME, clrGrowOut.length - 1), 0);
		int intFontIndex = StrictMath.max(StrictMath.min((fntGrowOut.length * intRunning) / GROW_OUT_TIME, fntGrowOut.length - 1), 0);
		g.setColor(clrGrow);
		// g.setColor(clrGrowOut[intColorIndex]);
		g.setFont(fntGrowOut[intFontIndex]);
		g.drawString(strGrow, intXGrowOut[intFontIndex], intYGrowOut[intFontIndex]);		
	    } 

	    g.setFont(fntTmp);
	    g.setColor(clrTmp);
	}
    }

    public class StringGrowIn extends StringAnimation {
	protected static final int GROW_TIME = 3000;
	protected static final int IN_TIME = 2000;

	protected final int intTime;
	protected final String strGrow;
	protected final Color clrGrow;
	protected final Font[] fntGrow;
	protected final int[] intX;
	protected final int[] intY;
	
	public StringGrowIn(String strGrow, Color clrGrow, Font fntGrow, int intOffset) {
	    super(intOffset);
	    this.intTime = GROW_TIME + IN_TIME;
	    this.strGrow = strGrow;
	    this.clrGrow = clrGrow;

	    final String strFamily = fntGrow.getFamily();
	    final int intStyle = fntGrow.getStyle();
	    final int intSize = fntGrow.getSize();

	    this.fntGrow = new Font[intSize];
	    this.intX = new int[intSize];
	    this.intY = new int[intSize];
	    for (int i = 0; i < intSize; i++) {
		this.fntGrow[i] = new Font(strFamily, intStyle, i);
		Point pntXY = getCenteredPosition(strGrow, this.fntGrow[i]);
		intX[i] = pntXY.x; intY[i] = pntXY.y;
	    }
	}

	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    int intIndex = StrictMath.max(StrictMath.min((fntGrow.length * intRunning) / GROW_TIME, fntGrow.length - 1), 0);
	    Color clrTmp = g.getColor();
	    Font fntTmp = g.getFont();
	    g.setColor(clrGrow);
	    g.setFont(fntGrow[intIndex]);	    

	    g.drawString(strGrow, intX[intIndex], intY[intIndex]);

	    g.setFont(fntTmp);
	    g.setColor(clrTmp);
	}
    }

    public class StringFadeIn extends StringAnimation {
	protected static final int FADE_TIME = 1500;
	protected static final int IN_TIME = 2000;

	protected final int intTime;
	protected final String strFade;
	protected final Color[] clrFade;
	protected final Font fntFade;
	protected final int intX;
	protected final int intY;

	public StringFadeIn(String strFade, Color clrFade, Font fntFade, int intOffset) {
	    super(intOffset);
	    this.intTime = FADE_TIME + IN_TIME;
	    this.strFade = strFade;
	    this.fntFade = fntFade;
	    Point pntXY = getCenteredPosition(strFade, fntFade);
	    this.intX = pntXY.x; this.intY = pntXY.y; 
	    
	    final int intR = clrFade.getRed();
	    final int intG = clrFade.getGreen();
	    final int intB = clrFade.getBlue();
	    final int intMaxAlpha = clrFade.getAlpha();
	    this.clrFade = new Color[intMaxAlpha];
	    for (int i = 0; i < intMaxAlpha - 1; i++) this.clrFade[i] = new Color(intR, intG, intB, i);
	    this.clrFade[intMaxAlpha - 1] = clrFade;
	}
	
	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    int intIndex = StrictMath.max(StrictMath.min((clrFade.length * intRunning) / FADE_TIME, clrFade.length - 1), 0);
	    Color clrTmp = g.getColor();
	    Font fntTmp = g.getFont();
	    g.setColor(clrFade[intIndex]);
	    g.setFont(fntFade);	    

	    g.drawString(strFade, intX, intY);

	    g.setFont(fntTmp);
	    g.setColor(clrTmp);
	}
    }
    	
    public class StringFallIn extends StringAnimation {
	protected static final int FALL_TIME = 2000;
	protected static final int IN_TIME = 2000;

	protected final int intTime;
	protected final String strFall;
	protected final Color clrFall;
	protected final Font fntFall;
	protected final int intX;
	protected final int intY;

	public StringFallIn(String strFall, Color clrFall, Font fntFall, int intOffset) {
	    super(intOffset);
	    this.intTime = FALL_TIME + IN_TIME;
	    this.strFall = strFall;
	    this.clrFall = clrFall;
	    this.fntFall = fntFall;

	    Point pntXY = getCenteredPosition(strFall, fntFall);
	    this.intX = pntXY.x; this.intY = pntXY.y; 
	}

	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    Color clrTmp = g.getColor();
	    Font fntTmp = g.getFont();
	    g.setColor(clrFall);
	    g.setFont(fntFall);

	    if (intRunning < FALL_TIME) {
		final int intFalled = (intY * intRunning) / FALL_TIME;
		g.drawString(strFall, intX, intFalled);
	    } else {
		g.drawString(strFall, intX, intY);
	    }		

	    g.setFont(fntTmp);
	    g.setColor(clrTmp);
	}
    }

    public class StringDropOut extends StringAnimation {
	protected static final int DROP_TIME = 1500;
	protected static final int DROP_DELAY = 300;

	protected final int intTime;
	protected final String strDrop;
	protected final Color clrDrop;
	protected final Font fntDrop;
	protected final int intX[];
	protected final int intY;
	protected final int intDrop;
	protected final int intDropDelay;
	protected final String[] strDroppingLetter;
	protected final int[] intDelay;

	protected final Random rand;
	
	public StringDropOut(String strDrop, Color clrDrop, Font fntDrop, int intOffset) {
	    super(intOffset);
	    this.strDrop = strDrop;
	    this.clrDrop = clrDrop;
	    this.fntDrop = fntDrop;

	    Point pntXY = getCenteredPosition(strDrop, fntDrop);
	    final int intX = pntXY.x; 
	    this.intY = pntXY.y; 
	    this.intDrop = this.intY;
	    this.intDropDelay = (intDrop * DROP_DELAY) / DROP_TIME;

	    final int intLength = strDrop.length();
	    this.strDroppingLetter = new String[intLength];
	    this.intX = new int[intLength];
	    this.intDelay = new int[intLength];
	    this.strDroppingLetter[0] = strDrop.substring(0, 1);
	    this.intX[0] = intX;
	    for (int i = 1; i < intLength; i++) {
		this.strDroppingLetter[i] = strDrop.substring(i, i + 1);
		this.intX[i] = intX + getOffset(strDrop, fntDrop, i);
	    }
	    this.intTime = DROP_TIME + (intLength - 1) * DROP_DELAY;

	    this.rand = new Random(System.currentTimeMillis());
	}

	public void init() {
	    super.init();
	    final int[] intOrder = generateRandomOrder(this.intDelay.length);
	    this.intDelay[intOrder[0]] = 0;
	    int intDelay = 0;
	    for (int i = 1; i < this.intDelay.length; i++) {
		intDelay += intDropDelay;
		this.intDelay[intOrder[i]] = intDelay;
	    }
	}

	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    Color clrTmp = g.getColor();
	    Font fntTmp = g.getFont();
	    g.setColor(clrDrop);
	    g.setFont(fntDrop);

	    final int intDropped = (intDrop * intRunning) / DROP_TIME;
	    for (int i = 0; i < strDroppingLetter.length; i++) {
		int intDelayDropped = StrictMath.max(0, intDropped - intDelay[i]);
		g.drawString(strDroppingLetter[i], intX[i], intY + intDelayDropped);
	    }

	    g.setFont(fntTmp);
	    g.setColor(clrTmp);
	}

	protected int[] generateRandomOrder(int intN) {
	    int[] intOrder = new int[intN];
	    for (int i = 0; i < intN; i++) intOrder[i] = i;

	    int[] intRandom = new int[intN];
	    int intOrdered = intN;
	    for (int i = 0; i < intN; i++) {
		int intRandomI = rand.nextInt(intOrdered);
		intRandom[i] = intOrder[intRandomI];
		intOrder[intRandomI] = intOrder[--intOrdered];
	    }
	    return intRandom;
	}
    }

    public class StringRotateOut extends StringAnimation {
	protected static final double A = StrictMath.PI / (1000.0);
	protected static final int DEFAULT_TIME = 6000;

	protected final int intTime;
	protected final String strRotate;
	protected final Color clrRotate;
	protected final Font[] fntRotate;
	protected final int[] intX;
	protected final int[] intY;
	protected final int intTranslateX;
	protected final int intTranslateY;

	public StringRotateOut(String strRotate, Color clrRotate, Font fntRotate, int intOffset) {
	    super(intOffset);
	    this.intTime = DEFAULT_TIME;
	    this.strRotate = strRotate;
	    this.clrRotate = clrRotate;

	    JComponent jpcRoot = anmThis.getRootPane();
	    Dimension dimRootSize = jpcRoot.getSize();
	    final int intRootWidth = dimRootSize.width;
	    final int intRootHeight = dimRootSize.height;
	    final int intMaxWidth = (intRootWidth * 9) / 10;
	    final String strFamily = fntRotate.getFamily();
	    final int intStyle = fntRotate.getStyle();
	    final int intSize = fntRotate.getSize();

	    this.fntRotate = new Font[intSize];	    
	    this.intX = new int[intSize];
	    this.intY = new int[intSize];
	    for (int i = 0; i < intSize; i++) {
		this.fntRotate[i] = new Font(strFamily, intStyle, i);
		Point pntXY = getCenteredPosition(strRotate, this.fntRotate[i]);
		intX[i] = pntXY.x; intY[i] = pntXY.y;
	    }				
	    this.intTranslateX = intRootWidth / 2;
	    this.intTranslateY = intRootHeight / 2;
	}
	
	public final int getRunningTime() { return intTime; }

	protected void paint2(Graphics g, int intRunning) {
	    Graphics2D g2d = (Graphics2D)g;
	    int intZ = StrictMath.max(StrictMath.min((fntRotate.length * (this.intTime - intRunning)) / this.intTime, fntRotate.length -1), 0);
	    Color clrTmp = g2d.getColor();
	    Font fntTmp = g2d.getFont();
	    AffineTransform aftTmp = g2d.getTransform();

	    g2d.setColor(clrRotate);
	    g2d.setFont(fntRotate[intZ]);	    
	    g2d.rotate(getTheta(intRunning, intZ), intTranslateX, intTranslateY);

	    g2d.drawString(strRotate, intX[intZ], intY[intZ]);	    
	    
	    g2d.setTransform(aftTmp);
	    g2d.setFont(fntTmp);
	    g2d.setColor(clrTmp);
	}

	protected final double getTheta(int intTimePassed, int intZ) {
	    double x = (double)intTimePassed;
	    double z = (double)(fntRotate.length - intZ);
	    double max_z = (double)fntRotate.length;
	    //	    double dblTheta = (A * (x + (z * x) / max_z)) % (2 * StrictMath.PI);
	    double dblTheta = (A * (x * z) / max_z) % (2 * StrictMath.PI);
	    //	    System.out.println("theta=" + dblTheta);
	    return dblTheta;
	}

    }
    


    private final Animator anmThis;

    public AnimationFactory(Animator anmThis) {
	this.anmThis = anmThis;
    }

    public Font getFontForWidth(String strTarget, Font fntProposed, int intWidthPercent) {
	final JComponent jpcRoot = anmThis.getRootPane();
	final int intMaxWidth = (jpcRoot.getWidth() * intWidthPercent) / 100;
	final String strFamily = fntProposed.getFamily();
	final int intStyle = fntProposed.getStyle();

	int intSize = fntProposed.getSize();
	Font fntCheck = fntProposed;
	FontMetrics fmtCheck = jpcRoot.getFontMetrics(fntCheck);
	int intStringWidth = fmtCheck.stringWidth(strTarget);
	while (intStringWidth < intMaxWidth) {
	    intSize += 10;
	    fntCheck = new Font(strFamily, intStyle, intSize);
	    fmtCheck = jpcRoot.getFontMetrics(fntCheck);
	    intStringWidth = fmtCheck.stringWidth(strTarget);
	}
	
	return fntCheck;
    }

    public StringAnimation createStringGrowInAndOut(String strGrow, Color clrGrow, Font fntGrow, int intOffset) {
	return new StringGrowInAndOut(strGrow, clrGrow, fntGrow, intOffset);
    }

    public StringAnimation createStringGrowIn(String strGrow, Color clrGrow, Font fntGrow, int intOffset) {
	return new StringGrowIn(strGrow, clrGrow, fntGrow, intOffset);
    }

    public StringAnimation createStringFallIn(String strFall, Color clrFall, Font fntFall, int intOffset) {
	return new StringFallIn(strFall, clrFall, fntFall, intOffset);
    }

    public StringAnimation createStringFadeIn(String strFade, Color clrFade, Font fntFade, int intOffset) {
	return new StringFadeIn(strFade, clrFade, fntFade, intOffset);
    }

    public StringAnimation createStringDropOut(String strDrop, Color clrDrop, Font fntDrop, int intOffset) {
	return new StringDropOut(strDrop, clrDrop, fntDrop, intOffset);
    }

    public StringAnimation createStringRotateOut(String strRotate, Color clrRotate, Font fntRotate, int intOffset) {
	return new StringRotateOut(strRotate, clrRotate, fntRotate, intOffset);
    }

    public Animator.Animation createFlyingLabels(JLabel[] jlbSource, JLabel jlbTarget) {
	return createFlyingLabels(jlbSource, jlbTarget, null, 0, DEFAULT_TIME);
    }


    public Animator.Animation createFlyingLabels(JLabel[] jlbSource, JLabel jlbTarget, Color clrTarget, int intOffset, int intTime) {
	String strText = null;
	boolean blnText = true;
	Font fntFont = null;
	boolean blnFont = true;
	int intCount = 0;
	for (int i = 0; i < jlbSource.length; i++) {
	    if (jlbSource[i] != null) if (jlbSource[i].getText() != null) {
		if (intCount == 0) {
		    strText = jlbSource[i].getText();
		    fntFont = jlbSource[i].getFont();
		} else {
		    if (!jlbSource[i].getText().equals(strText)) blnText = false;
		    if (!jlbSource[i].getFont().equals(fntFont)) blnFont = false;
		}
		intCount++;
	    }
	}
	if (intCount == 0) return null;

	boolean blnCommon = blnText && blnFont 
	    && jlbTarget.getText().equals(strText) && jlbTarget.getFont().equals(fntFont);
	JLabel[] jlbPure = new JLabel[intCount];
	int j = 0;
	for (int i = 0; i < jlbSource.length; i++) {
	    if (jlbSource[i] != null) if (jlbSource[i].getText() != null) jlbPure[j++] = jlbSource[i];
	}
	if (clrTarget == null) clrTarget = jlbPure[0].getForeground();
	if (blnCommon) {
	    //	    System.out.println("reunification case");
	    return new LabelReunification(jlbPure, jlbTarget, clrTarget, intOffset, anmThis, intTime);
	} else {
	    //	    System.out.println("unification case");
	    if (!blnFont) System.out.println("    warning: different source fonts");
	    return new LabelUnification(jlbPure, jlbTarget, clrTarget, intOffset, anmThis, intTime);
	}
    }
}
