/*
Animator
Copyright (C) 2003 Bj\"orn Hoffmeister

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package de.uni_luebeck.tcs.animation;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

public class Animator {
    private static final Integer SOURCE = JLayeredPane.DEFAULT_LAYER;
    private static final Integer ANIMATOR = new Integer(SOURCE.intValue() + 1);

    private static final int ANIMATOR_PRIORITY = Thread.NORM_PRIORITY - 1;

    public static final int IDLE = 0;
    public static final int RUNNING = 1;
    public static final int PAUSED = 2;
    public static final String[] STATE = {"IDLE", "RUNNING", "PAUSED"};

    private final int FPS = 32;
    private final long DELAY = (long)(1000 / FPS);
	
    public interface Listener {
	public void animationFinnished(Animation anmEnd) ;

	public void animatorFinnished();

	public void animatorStopped();
    }
	
    public interface Animation {
	// erst hier Komponenten dem AnimationsPane hinzufuegen
	public void init();

	// persistente Aenderungen vornehmen, die im weiteren Verlauf der
	// Animation aufgetreten waeren
	// Komponenten aus AnimationsPane entfernen ist nicht notwendig
	public void stop();

	// Komponenten im AnimationsPane verschieben und skalieren
	// und direkte Grafikoperationen im AnimationsPane
	public void paint(Graphics g, int intRunning);

	public boolean isFinnished();
    }

    private final class AnimatorPane extends JLayeredPane implements ComponentListener {
	public AnimatorPane() {
	    super();
	    add(jcpSource, SOURCE);
	    add(apeAnimator, ANIMATOR);	    
	    //	    apeAnimator.setSize(jcpSource.getSize());
	    //	    setSize(jcpSource.getSize());	    
	    addComponentListener(this);
	}

	public Dimension getPreferredSize() {
	    return jcpSource.getPreferredSize();
	}

	public Dimension getMaximumSize() {
	    return jcpSource.getMaximumSize();
	}

	public Dimension getMinimumSize() {
	    return jcpSource.getMinimumSize();
	}
	    
	// ComponentListener
	public void componentHidden(ComponentEvent e) {
	    //	    stop();
	}

	public void componentMoved(ComponentEvent e) {}

	public void componentResized(ComponentEvent e) {
	    //	    stop();
	    jcpSource.setSize(getSize());
	    apeAnimator.setSize(getSize());
	    jcpSource.revalidate();
	}

	public void componentShown(ComponentEvent e) {}
    }

    public final class AnimationPane extends JPanel implements MouseListener {
	protected long lngStarted = -1;

	private final Font fntDefault;
	private final Font fntStatus;

	// state
	private final int intStateWidth, intStateHeight;
	protected boolean blnState = true;
	// fps
	private final int intFPSWidth, intFPSHeight;
	protected boolean blnFPS = true;
	protected float fltFPS = 0.0f;
	protected String strFPS = "0.0fps";
	protected int intRepaint = 0;

	protected AnimationPane(boolean blnIsInterruptable, boolean blnFPS) {
	    super(null);
	    this.blnFPS = blnFPS;
	    if (blnIsInterruptable) addMouseListener(this);
	    setOpaque(false);
	    setVisible(false);

	    fntDefault = getFont();
	    fntStatus = new Font(fntDefault.getFamily(), Font.PLAIN, 10);
	    FontMetrics fmtThis = getFontMetrics(fntStatus);
	    intStateHeight = fmtThis.getHeight() + 1;
	    intStateWidth = fmtThis.stringWidth("RUNNING") + 1;
	    intFPSHeight = fmtThis.getHeight() + 1;
	    intFPSWidth = fmtThis.stringWidth("188.8fps") + 1;
	}

	protected void clear() {
	    removeAll();
	}
	
	public void paintComponent(Graphics g) {
	    if (intState == RUNNING) {
		int intRunning = getRunningTime();
		for (int i = 0; i < altAnimation.size(); i++) {
		    Animation anmThis = (Animation)altAnimation.get(i);
		    if (!anmThis.isFinnished()) {
			anmThis.paint(g, intRunning);
			if (anmThis.isFinnished()) finnished(anmThis);		    
		    }
		}
		// fps
		if (blnFPS) paintFPS(g);
	    }

	    // state
	    if (blnState) paintState(g);
	}

	public void paintBorder(Graphics g) {}

	private int getRunningTime() {
	    if (lngStarted == -1) {
		return 0;
	    } else {
		return (int)(System.currentTimeMillis() - lngStarted);
	    }
	}

	private void paintState(Graphics g) {
	    g.setFont(fntStatus);
	    g.drawString(STATE[intState], getWidth() - intStateWidth, intStateHeight);
	    g.setFont(fntDefault);
	}
	
	private void paintFPS(Graphics g) {
	    g.setFont(fntStatus);
	    g.drawString(strFPS, getWidth() - intFPSWidth, ((blnState)?intFPSHeight + intStateHeight:intFPSHeight));
	    g.setFont(fntDefault);
	    intRepaint++;	    
	}
	
	// MouseListener
	public void mouseClicked(MouseEvent e) {
	    if (e.getClickCount() > 1) stop();
	    /*
	    else switch (intState) {
	    case PAUSED:
		resume();
		break;
	    default:
		pause();
	    }
	    */
	}

	public void mouseEntered(MouseEvent e) {}

	public void mouseExited(MouseEvent e) {}

	public void mousePressed(MouseEvent e) {}

	public void mouseReleased(MouseEvent e) {}
    }

    private final class AnimatorThread extends Thread {	
	final long lngDelay;
	long lngNext = 0;

	// trigger
	boolean blnAlive = true;

	// fps
	long lngLastUpdate;

	public AnimatorThread(long lngDelay) {
	    this.lngDelay = lngDelay;
	}

	public void run() {
	    long lngNow = 0;

	    while (blnAlive) {
		switch (intState) {
		case IDLE:
		    //		    System.out.println("IDL");

		    pause();
		    lngNow = System.currentTimeMillis();    
		    lngNext = lngNow + lngDelay;		    
		    apeAnimator.lngStarted = lngNow;		    
		    break;
		case RUNNING:
		    lngNow = System.currentTimeMillis();    
		    //		    System.out.println("RUN: " + lngNow);

		    // frames count
		    if (Animator.this.apeAnimator.blnFPS) frameRate(lngNow);
		    // animate
		    Animator.this.apeAnimator.repaint();
		    // sleep
		    long lngSleep = lngNext - lngNow;
		    if (lngSleep > 0) {
			try{ sleep(lngSleep); } catch(InterruptedException e) {}
			lngNow = System.currentTimeMillis();
		    }
		    lngNext = lngNow + lngDelay;		    
		    break;
		case PAUSED:
		    //		    System.out.println("PAU");

		    Animator.this.apeAnimator.repaint();		    
		    long lngBegin = System.currentTimeMillis();
		    pause();
		    apeAnimator.lngStarted += (System.currentTimeMillis() - lngBegin);
		    break;
		}
	    }
	}

	private synchronized void pause() {
	    try{ wait(); } catch(InterruptedException e) {}
	}

	protected synchronized void resumeThread() {
	    notifyAll();
	}

	private void frameRate(long lngNow) {
	    long lngGone = lngNow - lngLastUpdate;
	    if (lngGone > 250) {
		apeAnimator.fltFPS = (float)(apeAnimator.intRepaint * 1000) / (float)lngGone;
	        apeAnimator.intRepaint = 0;
		lngLastUpdate = lngNow;

		int intPrint = (int)(apeAnimator.fltFPS * 10.0f);
		apeAnimator.strFPS = (intPrint / 10) + "." + intPrint % 10 + "fps";
	    }
	}
    }

    private final ArrayList altAnimation;
    private final ArrayList altListener;
    private final JComponent jcpSource;
    private final AnimationPane apeAnimator;
    private final AnimatorPane arpAnimator;

    private AnimatorThread athAnimator;
    private int intAnimationCount;
    private int intState = IDLE;
    
    public Animator(JComponent jcpSource, boolean blnInterruptable) {	    
	this.altAnimation = new ArrayList(16);
	this.altListener = new ArrayList(4);

	this.jcpSource = jcpSource;
	this.apeAnimator = new AnimationPane(blnInterruptable, true);
	this.arpAnimator = new AnimatorPane();

	this.athAnimator = new AnimatorThread(DELAY);
	this.athAnimator.setPriority(ANIMATOR_PRIORITY);
	this.athAnimator.setDaemon(true);
	this.athAnimator.start();
    }

    public JComponent getRootPane() {
	return jcpSource;
    }

    public JComponent getAnimatorPane() {
	return (JComponent)arpAnimator;
    }

    public JComponent getAnimationPane() {
	return (JComponent)apeAnimator;
    }

    public int getState() {
	return intState;
    }

    //	public Animation addFlyingLabelsAnimation(JLabel[] jlbSource, JLabel jlbTarget, Container cntParent) {}

    // add animation
    public void add(Animation anmAdd) {
	altAnimation.add(anmAdd);
    }

    /*
      public void removeAnimation(Animation anmRemove) {
      if (intState == STARTED) return;
      int intPos = altAnimation.indexOf(anmRemove);
      return (intPos == -1)?null:altAnimation.remove(intPos);
      }
    */

    public void removeAll() {
	if (intState == RUNNING) return;
	altAnimation.clear();
    }

    // start und stop
    public void start() {
	if (intState != IDLE) return;
	intAnimationCount = altAnimation.size();
	for (int i = 0; i < altAnimation.size(); i++) {
	    ((Animation)altAnimation.get(i)).init();
	}

	apeAnimator.setVisible(true);
	intState = RUNNING;
	athAnimator.resumeThread();
    }

    public void stop() {
	if (intState == IDLE) return;
	intState = IDLE;
	athAnimator.interrupt();

	apeAnimator.setVisible(false);
	for (int i = 0; i < altAnimation.size(); i++) {
	    Animation anmStop = (Animation)altAnimation.get(i);
	    if (!anmStop.isFinnished()) anmStop.stop();
	}
	clearAnimator();
	fireAnimatorStopped();
    }

    // pause
    public void pause() {
	if (intState != RUNNING) return;
	intState = PAUSED;
	athAnimator.interrupt();
    }

    public void resume() {
	if (intState != PAUSED) return;
	intState = RUNNING;
	athAnimator.resumeThread();
    }

    // finnish
    private void finnished(Animation anmFinnish) {
	fireAnimationFinnished(anmFinnish);
	if (--intAnimationCount == 0) finnishedAll();
    }

    private void finnishedAll() {
	intState = IDLE;
	athAnimator.interrupt();

	apeAnimator.setVisible(false);
	clearAnimator();
	fireAnimatorFinnished();
    }	    
    
    private void clearAnimator() {
	apeAnimator.removeAll();
	altAnimation.clear();
	arpAnimator.repaint();
    }



    // AnimatorListener
    public void addAnimatorListener(Animator.Listener alrAdd) {
	altListener.add(alrAdd);
    }

    public Animator.Listener removeAnimatorListener(Animator.Listener alrRemove) {
	int intPos = altListener.indexOf(alrRemove);
	return (intPos == -1)?null:(Animator.Listener)altListener.remove(intPos);
    }
	
    public void removeAllAnimatorListener() {
	altListener.clear();
    }
	
    private void fireAnimationFinnished(Animation anmFinnish) {
	for (int i = 0; i < altListener.size(); i++) {
	    ((Animator.Listener)altListener.get(i)).animationFinnished(anmFinnish);
	}
    }

    private void fireAnimatorFinnished() {
	for (int i = 0; i < altListener.size(); i++) {
	    ((Animator.Listener)altListener.get(i)).animatorFinnished();
	}
    }

    private void fireAnimatorStopped() {
	for (int i = 0; i < altListener.size(); i++) {
	    ((Animator.Listener)altListener.get(i)).animatorStopped();
	}
    }
}

